/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.jcs.auxiliary.disk.block;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.jcs.engine.behavior.IElementSerializer;
import org.apache.commons.jcs.utils.serialization.StandardSerializer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class BlockDisk {
    private static final Log log = LogFactory.getLog(BlockDisk.class);
    public static final byte HEADER_SIZE_BYTES = 4;
    private static final int DEFAULT_BLOCK_SIZE_BYTES = 4096;
    private final int blockSizeBytes;
    private final AtomicInteger numberOfBlocks = new AtomicInteger(0);
    private final ConcurrentLinkedQueue<Integer> emptyBlocks = new ConcurrentLinkedQueue();
    private final IElementSerializer elementSerializer;
    private final String filepath;
    private final FileChannel fc;
    private final AtomicLong putBytes = new AtomicLong(0L);
    private final AtomicLong putCount = new AtomicLong(0L);

    public BlockDisk(File file, IElementSerializer elementSerializer) throws IOException {
        this(file, 4096, elementSerializer);
    }

    public BlockDisk(File file, int blockSizeBytes) throws IOException {
        this(file, blockSizeBytes, new StandardSerializer());
    }

    public BlockDisk(File file, int blockSizeBytes, IElementSerializer elementSerializer) throws IOException {
        this.filepath = file.getAbsolutePath();
        RandomAccessFile raf = new RandomAccessFile(this.filepath, "rw");
        this.fc = raf.getChannel();
        this.numberOfBlocks.set((int)Math.ceil(1.0f * (float)this.fc.size() / (float)blockSizeBytes));
        if (log.isInfoEnabled()) {
            log.info("Constructing BlockDisk, blockSizeBytes [" + blockSizeBytes + "]");
        }
        this.blockSizeBytes = blockSizeBytes;
        this.elementSerializer = elementSerializer;
    }

    private int[] allocateBlocks(int numBlocksNeeded) {
        assert (numBlocksNeeded >= 1);
        int[] blocks = new int[numBlocksNeeded];
        for (int i2 = 0; i2 < numBlocksNeeded; ++i2) {
            Integer emptyBlock = this.emptyBlocks.poll();
            if (emptyBlock == null) {
                emptyBlock = this.numberOfBlocks.getAndIncrement();
            }
            blocks[i2] = emptyBlock;
        }
        return blocks;
    }

    protected int[] write(Serializable object) throws IOException {
        byte[] data = this.elementSerializer.serialize(object);
        if (log.isDebugEnabled()) {
            log.debug("write, total pre-chunking data.length = " + data.length);
        }
        this.putBytes.addAndGet(data.length);
        this.putCount.incrementAndGet();
        int numBlocksNeeded = this.calculateTheNumberOfBlocksNeeded(data);
        if (log.isDebugEnabled()) {
            log.debug("numBlocksNeeded = " + numBlocksNeeded);
        }
        int[] blocks = this.allocateBlocks(numBlocksNeeded);
        int offset = 0;
        int maxChunkSize = this.blockSizeBytes - 4;
        ByteBuffer headerBuffer = ByteBuffer.allocate(4);
        for (int i2 = 0; i2 < numBlocksNeeded; ++i2) {
            headerBuffer.clear();
            int length = Math.min(maxChunkSize, data.length - offset);
            headerBuffer.putInt(length);
            ByteBuffer dataBuffer = ByteBuffer.wrap(data, offset, length);
            long position = this.calculateByteOffsetForBlockAsLong(blocks[i2]);
            headerBuffer.flip();
            int written = this.fc.write(headerBuffer, position);
            assert (written == 4);
            written = this.fc.write(dataBuffer, position + 4L);
            assert (written == length);
            offset += length;
        }
        return blocks;
    }

    protected byte[][] getBlockChunks(byte[] complete, int numBlocksNeeded) {
        byte[][] chunks = new byte[numBlocksNeeded][];
        if (numBlocksNeeded == 1) {
            chunks[0] = complete;
        } else {
            int maxChunkSize = this.blockSizeBytes - 4;
            int totalBytes = complete.length;
            int totalUsed = 0;
            for (int i2 = 0; i2 < numBlocksNeeded; i2 = (int)((short)(i2 + 1))) {
                int chunkSize = Math.min(maxChunkSize, totalBytes - totalUsed);
                byte[] chunk = new byte[chunkSize];
                System.arraycopy(complete, totalUsed, chunk, 0, chunkSize);
                chunks[i2] = chunk;
                totalUsed += chunkSize;
            }
        }
        return chunks;
    }

    protected <T extends Serializable> T read(int[] blockNumbers) throws IOException, ClassNotFoundException {
        byte[] data = null;
        if (blockNumbers.length == 1) {
            data = this.readBlock(blockNumbers[0]);
        } else {
            ByteArrayOutputStream baos = new ByteArrayOutputStream(this.getBlockSizeBytes());
            for (int i2 = 0; i2 < blockNumbers.length; i2 = (int)((short)(i2 + 1))) {
                byte[] chunk = this.readBlock(blockNumbers[i2]);
                baos.write(chunk);
            }
            data = baos.toByteArray();
            baos.close();
        }
        if (log.isDebugEnabled()) {
            log.debug("read, total post combination data.length = " + data.length);
        }
        return (T)((Serializable)this.elementSerializer.deSerialize(data, null));
    }

    private byte[] readBlock(int block) throws IOException {
        int datalen = 0;
        String message = null;
        boolean corrupted = false;
        long fileLength = this.fc.size();
        long position = this.calculateByteOffsetForBlockAsLong(block);
        ByteBuffer datalength = ByteBuffer.allocate(4);
        this.fc.read(datalength, position);
        datalength.flip();
        datalen = datalength.getInt();
        if (position + (long)datalen > fileLength) {
            corrupted = true;
            message = "Record " + position + " exceeds file length.";
        }
        if (corrupted) {
            log.warn("\n The file is corrupt: \n " + message);
            throw new IOException("The File Is Corrupt, need to reset");
        }
        ByteBuffer data = ByteBuffer.allocate(datalen);
        this.fc.read(data, position + 4L);
        data.flip();
        return data.array();
    }

    protected void freeBlocks(int[] blocksToFree) {
        if (blocksToFree != null) {
            for (int i2 = 0; i2 < blocksToFree.length; i2 = (int)((short)(i2 + 1))) {
                this.emptyBlocks.offer(blocksToFree[i2]);
            }
        }
    }

    @Deprecated
    protected int calculateByteOffsetForBlock(int block) {
        return block * this.blockSizeBytes;
    }

    protected long calculateByteOffsetForBlockAsLong(int block) {
        return (long)block * (long)this.blockSizeBytes;
    }

    protected int calculateTheNumberOfBlocksNeeded(byte[] data) {
        int dataLength = data.length;
        int oneBlock = this.blockSizeBytes - 4;
        if (dataLength <= oneBlock) {
            return 1;
        }
        int dividend = dataLength / oneBlock;
        if (dataLength % oneBlock != 0) {
            ++dividend;
        }
        return dividend;
    }

    protected long length() throws IOException {
        return this.fc.size();
    }

    protected void close() throws IOException {
        this.fc.close();
    }

    protected synchronized void reset() throws IOException {
        this.numberOfBlocks.set(0);
        this.emptyBlocks.clear();
        this.fc.truncate(0L);
        this.fc.force(true);
    }

    protected int getNumberOfBlocks() {
        return this.numberOfBlocks.get();
    }

    protected int getBlockSizeBytes() {
        return this.blockSizeBytes;
    }

    protected long getAveragePutSizeBytes() {
        long count = this.putCount.get();
        if (count == 0L) {
            return 0L;
        }
        return this.putBytes.get() / count;
    }

    protected int getEmptyBlocks() {
        return this.emptyBlocks.size();
    }

    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append("\nBlock Disk ");
        buf.append("\n  Filepath [" + this.filepath + "]");
        buf.append("\n  NumberOfBlocks [" + this.numberOfBlocks.get() + "]");
        buf.append("\n  BlockSizeBytes [" + this.blockSizeBytes + "]");
        buf.append("\n  Put Bytes [" + this.putBytes + "]");
        buf.append("\n  Put Count [" + this.putCount + "]");
        buf.append("\n  Average Size [" + this.getAveragePutSizeBytes() + "]");
        buf.append("\n  Empty Blocks [" + this.getEmptyBlocks() + "]");
        try {
            buf.append("\n  Length [" + this.length() + "]");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return buf.toString();
    }

    protected String getFilePath() {
        return this.filepath;
    }
}

